/* Contain the entire script within a function because REPORTER only has a single JavaScript realm
 * for the entire session. */
chest_score();

/**
 * Performs the chest score calculation for the specified models
 */
function chest_score() {
    let template = Template.GetCurrent();
    let models = ["M1", "T1"];
    let body_region_label = "chest";

    let occupants = ["DRIVER"];

    for (let m of models) {
        for (let occ of occupants) {
            LogPrint(`Calculating ${m} ${occ} ${body_region_label} score...`);

            /* Create a status object to track whether REPORTER Variables are all present and valid.
             * <success> is initially true but will be set to false if anything missing or invalid. */
            let status = { success: true, missing: [], invalid: [] };

            /* Get the head excursion zone and whether there's a countermeasure or not - required to set the high score value */
            let head_excursion_zone = get_variable_value(status, `${m}_${occ}_DRIVER_HEAD_EXCURSION_RATING`, "string");
            let countermeasure_int = get_variable_value(status, `COUNTERMEASURE`, "string");
            if (countermeasure_int === "Yes") {
                countermeasure_int = 1;
            } else {
                countermeasure_int = 0;
            }

            /* comment why checking for countermeasure_int === 1 */
            let countermeasure = countermeasure_int === 1;

            let CHEST_HI_SCORE = get_max_chest_score(head_excursion_zone, countermeasure);

            new Variable(
                template,
                `${m}_CHEST_HI_SCORE`,
                `Max chest score for head excursion in ${head_excursion_zone} zone with${
                    countermeasure ? "" : "out"
                } countermeasure.`,
                CHEST_HI_SCORE.toString(),
                "General",
                false,
                true
            );

            let CHEST_COMPRESSION_HI_LIMIT = 31; // The compression limit for the chest in mm
            let CHEST_COMPRESSION_LO_LIMIT = 41; // The compression limit for the chest in mm
            let CHEST_COMPRESSION_CAPPING_LIMIT = 41; // Capping limit is 50mm (same as lower limit)

            //set perfomance limits
            let CHEST_COMPRESSION_GOOD = CHEST_COMPRESSION_HI_LIMIT; //higher performance limit
            let CHEST_COMPRESSION_WEAK = CHEST_COMPRESSION_LO_LIMIT; //lower performance limit
            let CHEST_TOP_HI_SCORE = CHEST_HI_SCORE;
            let CHEST_TOP_LO_SCORE = 0;
            let CHEST_MID_HI_SCORE = CHEST_HI_SCORE;
            let CHEST_MID_LO_SCORE = 0;
            let CHEST_BOTTOM_HI_SCORE = CHEST_HI_SCORE;
            let CHEST_BOTTOM_LO_SCORE = 0;
            let max_upp_rib_com = get_variable_value(status, `${m}_${occ}_CHEST_COMPRESSION_UPPER_MAX_VALUE`, "float");
            let max_mid_rib_com = get_variable_value(status, `${m}_${occ}_CHEST_COMPRESSION_MIDDLE_MAX_VALUE`, "float");
            let max_bottom_rib_com = get_variable_value(
                status,
                `${m}_${occ}_CHEST_COMPRESSION_LOWER_MAX_VALUE`,
                "float"
            );

            if (max_upp_rib_com == "Missing" || max_upp_rib_com == null) {
                var upper_rib_compression_score = 0;
            } else {
                var upper_rib_compression_score = sliding_scale(
                    max_upp_rib_com,
                    CHEST_COMPRESSION_GOOD,
                    CHEST_COMPRESSION_WEAK,
                    CHEST_TOP_HI_SCORE,
                    CHEST_TOP_LO_SCORE
                );
            }

            if (max_mid_rib_com == "Missing" || max_mid_rib_com == null) {
                var mid_rib_compression_score = 0;
            } else {
                var mid_rib_compression_score = sliding_scale(
                    max_mid_rib_com,
                    CHEST_COMPRESSION_GOOD,
                    CHEST_COMPRESSION_WEAK,
                    CHEST_MID_HI_SCORE,
                    CHEST_MID_LO_SCORE
                );
            }

            if (max_bottom_rib_com == "Missing" || max_bottom_rib_com == null) {
                var bottom_rib_compression_score = 0;
            } else {
                var bottom_rib_compression_score = sliding_scale(
                    max_bottom_rib_com,
                    CHEST_COMPRESSION_GOOD,
                    CHEST_COMPRESSION_WEAK,
                    CHEST_BOTTOM_HI_SCORE,
                    CHEST_BOTTOM_LO_SCORE
                );
            }

            new Variable(
                template,
                `${m}_${occ}_CHEST_COMPRESSION_MAX_01_SCORE`,
                `Result from Automotive Assessments Workflow`,
                upper_rib_compression_score.toFixed(3),
                "General",
                false,
                true
            );
            new Variable(
                template,
                `${m}_${occ}_CHEST_COMPRESSION_MAX_02_SCORE`,
                `Result from Automotive Assessments Workflow`,
                mid_rib_compression_score.toFixed(3),
                "General",
                false,
                true
            );
            new Variable(
                template,
                `${m}_${occ}_CHEST_COMPRESSION_MAX_03_SCORE`,
                `Result from Automotive Assessments Workflow`,
                bottom_rib_compression_score.toFixed(3),
                "General",
                false,
                true
            );
            let maxc = Math.max(max_upp_rib_com, max_mid_rib_com, max_bottom_rib_com);
            new Variable(
                template,
                `${m}_CHEST_MAX_COMPRESSION`,
                `Result from Automotive Assessments Workflow`,
                maxc.toFixed(3),
                "General",
                false,
                true
            );
            let chest_score = Math.min(
                upper_rib_compression_score,
                mid_rib_compression_score,
                bottom_rib_compression_score
            );
            new Variable(
                template,
                `${m}_CHEST_MAX_COMPRESSION_SCORE`,
                `Result from Automotive Assessments Workflow`,
                chest_score.toFixed(3),
                "General",
                false,
                true
            );

            new Variable(
                template,
                `${m}_CHEST_CAPPING_LIMIT`,
                `Result from Automotive Assessments Workflow`,
                maxc >= CHEST_COMPRESSION_CAPPING_LIMIT ? "TRUE" : "FALSE",
                "General",
                false,
                true
            );
        }
    }
}

/**
 * Returns the maximum chest score based on the head excursion zone and countermeasure
 * @param {string} zone The head excursion zone
 * @param {boolean} countermeasure Whether there is a countermeasure or not
 * @returns {number} The maximum chest score
 */
function get_max_chest_score(zone, countermeasure) {
    if (countermeasure) {
        switch (zone) {
            case "CAPPING":
                return 0;
            case "RED":
            case "RED**":
                return 2;
            case "ORANGE":
                return 3;
            case "YELLOW":
                return 4;
            case "GREEN":
                return 4;
            default:
                return 0;
        }
    } else {
        switch (zone) {
            case "CAPPING":
                return 0;
            case "RED":
                return 1;
            case "ORANGE":
                return 1;
            case "YELLOW":
                return 2;
            case "GREEN":
                return 4;
            default:
                return 0;
        }
    }
}

function sliding_scale(val, hi_perf, lo_perf, hi_score, lo_score) {
    var retval = 0.0;

    if (val < hi_perf) retval = hi_score;
    else if (val > lo_perf) retval = lo_score;
    else retval = hi_score + ((val - hi_perf) * (lo_score - hi_score)) / (lo_perf - hi_perf);

    return retval;
}
